# -*- coding: utf-8 -*-
"""
@file: sched_example.py
@brief: 工位预约示例
@copyright: Copyright (C) 2024 ROKAE (Beijing) Technology Co., LTD. All Rights Reserved.
Information in this file is the intellectual property of Rokae Technology Co., Ltd,
And may contains trade secrets that must be stored and viewed confidentially.
"""
import setup_path
import platform
# 根据操作系统导入相应的模块
if platform.system() == 'Windows':
    from Release.windows import xCoreSDK_python
    from Release.windows.xCoreSDK_python.EventInfoKey import SchedState
elif platform.system() == 'Linux':
    from Release.linux import xCoreSDK_python
    from Release.linux.xCoreSDK_python.EventInfoKey import SchedState
else:
    raise ImportError("Unsupported operating system")
from log import print_log, print_separator


def main(robot: xCoreSDK_python.BaseRobot, ec: dict):
    # 设置回调/监视
    def printSchedInfo(info):
        enable = info[SchedState.Enable]
        registerEnable = info[SchedState.RegisterEnable]
        schedState = info[SchedState.Status]
        print(f"Enable: {enable}")
        print(f"RegisterEnable: {registerEnable}")
        print("Status:")
        for item in schedState:
            print(f"id: {item.id}")
            print(f"state: {item.state} rank: {item.rank} count: {item.count}")

    robot.setEventWatcher(xCoreSDK_python.Event.schedState, printSchedInfo, ec)
    print(f"setEventWatcher: {ec}")

    # 启用/停用预约
    enable = True
    robot.enableSched(enable, ec)
    print(f"enable sched: {ec}")
    enable_status = robot.getSchedEnableStatus(ec)
    print(f"getSchedEnableStatus: {ec}")
    print(f"sched enable status: {enable_status}")

    enable = False
    robot.enableSched(enable, ec)
    print(f"enable sched: {ec}")
    enable_status = robot.getSchedEnableStatus(ec)
    print(f"getSchedEnableStatus: {ec}")
    print(f"sched enable status: {enable_status}")

    # 开始/停止预约
    # 需要在启用预约，自动模式，上电之后才能开始预约
    robot.enableSched(True, ec)
    robot.setOperateMode(xCoreSDK_python.OperateMode.automatic, ec)
    robot.setPowerState(True, ec)

    registerEnable = True
    robot.registerEnableSched(registerEnable, ec)
    print(f"registerEnableSched: {ec}")
    reg_status = robot.getSchedRegisterEnableStatus(ec)
    print(f"getSchedRegisterEnableStatus: {ec}")
    print(f"sched register enable status: {reg_status}")

    registerEnable = False
    robot.registerEnableSched(registerEnable, ec)
    print(f"registerEnableSched: {ec}")
    reg_status = robot.getSchedRegisterEnableStatus(ec)
    print(f"getSchedRegisterEnableStatus: {ec}")
    print(f"sched register enable status: {reg_status}")

    # 预约设置
    # 可以使用projectsInfo获取控制器工程信息
    proj_info = robot.projectsInfo(ec)
    for info in proj_info:
        print(f"project name: {info.name}")

    sched_setting = xCoreSDK_python.SchedSetting()
    id = 2
    sched_setting.schedEntrys[id - 1].enable = True
    sched_setting.schedEntrys[id - 1].enable = True
    sched_setting.schedEntrys[id - 1].name = proj_info[0].name
    sched_setting.schedEntrys[id - 1].count = 3
    sched_setting.schedEntrys[id - 1].register_binding.type = "DI"
    sched_setting.schedEntrys[id - 1].register_binding.name = "DI0_0"
    sched_setting.schedEntrys[id - 1].register_cancel.type = 'DI'
    sched_setting.schedEntrys[id - 1].register_cancel.name = 'DI0_1'
    sched_setting.schedEntrys[id - 1].register_status.type = 'DO'
    sched_setting.schedEntrys[id - 1].register_status.name = 'DO0_0'

    robot.setSched(sched_setting, ec)
    print(f"setSched: {ec}")
    sched_setting_return = robot.getSchedSetting(ec)
    print(f"getSchedSetting: {ec}")

    # 打印调度设置
    for entry in sched_setting_return.schedEntrys:
        print(f"id: {entry.id}")
        print(
            f"enable: {entry.enable} name: {entry.name} count: {entry.count}")
        print(
            f"register type: {entry.register_binding.type} name: {entry.register_binding.name}"
        )
        print(
            f"register_cancel type: {entry.register_cancel.type} name: {entry.register_cancel.name}"
        )
        print(
            f"register_status type: {entry.register_status.type} name: {entry.register_status.name}"
        )

    # 预约/取消预约
    # 需要在开始预约之后才能预约
    robot.enableSched(True, ec)
    robot.setOperateMode(xCoreSDK_python.OperateMode.automatic, ec)
    robot.setPowerState(True, ec)
    robot.registerEnableSched(True, ec)

    def print_sched_status(status: list[xCoreSDK_python.SchedStatus]):
        for item in status:
            print(f"id: {item.id}")
            print(f"state: {item.state} rank: {item.rank} count: {item.count}")

    sched_status = robot.getSchedStatus(ec)
    print(f"getSchedStatus: {ec}")
    print_sched_status(sched_status)

    robot.unregisterSched(id, ec)
    print(f"unregisterSched: {ec}")
    sched_status = robot.getSchedStatus(ec)
    print(f"getSchedStatus: {ec}")
    print_sched_status(sched_status)

    robot.unregisterAllSched(ec)
    print(f"unregisterAllSched: {ec}")

    robot.clearSchedCount(ec)
    print(f"clearSchedCount: {ec}")

    robot.schedAdvancedSetting("pause_project", ec)
    print(f"schedAdvancedSetting: {ec}")
    robot.schedAdvancedSetting("unregister", ec)
    print(f"schedAdvancedSetting: {ec}")


if __name__ == "__main__":
    try:
        ip = "192.168.110.129"
        robot = xCoreSDK_python.xMateRobot(ip)
        ec = {}
        main(robot, ec)
        robot.setNoneEventWatcher(xCoreSDK_python.Event.schedState, ec)
    except Exception as e:
        print(f"Error: {str(e)}")
